"use client";

import { useEffect, useState } from "react";
import { api, withAuth } from "@/lib/apiClient";
import { getAdminToken } from "@/lib/auth";

type Payout = {
  id: string;
  hostId: string;
  amountNpr: number;
  method: string;
  details?: string | null;
  status: string;
  createdAt: string;
  updatedAt: string;
  host?: { fullName: string; email: string };
};

export default function PayoutsPage() {
  const [payouts, setPayouts] = useState<Payout[]>([]);
  const [error, setError] = useState<string | null>(null);
  const [loading, setLoading] = useState(true);

  const load = async () => {
    try {
      setLoading(true);
      setError(null);
      const token = getAdminToken();
      if (!token) {
        window.location.href = "/login";
        return;
      }
      withAuth();
      const res = await api.get("/payouts/admin");
      setPayouts(res.data);
    } catch {
      setError("Failed to load payouts");
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    load();
  }, []);

  const handleApprove = async (id: string) => {
    if (!confirm("Approve this payout?")) return;
    try {
      withAuth();
      await api.post(`/payouts/admin/${id}/approve`);
      await load();
    } catch {
      alert("Approve failed");
    }
  };

  const handleReject = async (id: string) => {
    const reason = prompt("Reject reason?");
    try {
      withAuth();
      await api.post(`/payouts/admin/${id}/reject`, { reason });
      await load();
    } catch {
      alert("Reject failed");
    }
  };

  return (
    <div>
      <h1>Payout Requests</h1>
      {error && <div style={{ color: "red" }}>{error}</div>}
      {loading ? (
        <div>Loading...</div>
      ) : payouts.length === 0 ? (
        <div>No payout requests.</div>
      ) : (
        <table style={{ marginTop: 16 }}>
          <thead>
            <tr>
              <th>ID</th>
              <th>Host</th>
              <th>Amount (NPR)</th>
              <th>Method</th>
              <th>Status</th>
              <th>Created</th>
              <th>Actions</th>
            </tr>
          </thead>
          <tbody>
            {payouts.map((p) => (
              <tr key={p.id}>
                <td>{p.id.slice(0, 8)}</td>
                <td>
                  {p.host?.fullName || p.hostId}
                  <br />
                  <span style={{ fontSize: 12, color: "#6b7280" }}>
                    {p.host?.email}
                  </span>
                </td>
                <td>{p.amountNpr}</td>
                <td>{p.method}</td>
                <td>{p.status}</td>
                <td>{new Date(p.createdAt).toLocaleString()}</td>
                <td>
                  {p.status === "PENDING" ? (
                    <>
                      <button onClick={() => handleApprove(p.id)} style={{ marginRight: 6 }}>
                        Approve
                      </button>
                      <button onClick={() => handleReject(p.id)}>Reject</button>
                    </>
                  ) : (
                    "-"
                  )}
                </td>
              </tr>
            ))}
          </tbody>
        </table>
      )}
    </div>
  );
}
