"use client";

import { useEffect, useState } from "react";
import { api, withAuth } from "@/lib/apiClient";
import { useRouter } from "next/navigation";
import { getAdminToken } from "@/lib/auth";

type DashboardData = {
  totalUsers: number;
  totalHosts: number;
  totalProperties: number;
  totalBookings: number;
  platformEarningsNpr: number;
};

export default function DashboardPage() {
  const router = useRouter();
  const [data, setData] = useState<DashboardData | null>(null);
  const [error, setError] = useState<string | null>(null);

  useEffect(() => {
    const token = getAdminToken();
    if (!token) {
      router.push("/login");
      return;
    }
    withAuth();
    api
      .get("/admin/dashboard")
      .then((res) => setData(res.data))
      .catch(() => setError("Failed to load dashboard"));
  }, [router]);

  if (error) return <div>{error}</div>;
  if (!data) return <div>Loading dashboard...</div>;

  return (
    <div>
      <h1>Dashboard</h1>
      <div
        style={{
          display: "grid",
          gridTemplateColumns: "repeat(auto-fit, minmax(160px, 1fr))",
          gap: 12,
          marginTop: 16
        }}
      >
        <StatCard title="Total Users" value={data.totalUsers} />
        <StatCard title="Total Hosts" value={data.totalHosts} />
        <StatCard title="Total Properties" value={data.totalProperties} />
        <StatCard title="Total Bookings" value={data.totalBookings} />
        <StatCard title="Platform Earnings (NPR)" value={data.platformEarningsNpr} />
      </div>
    </div>
  );
}

function StatCard({ title, value }: { title: string; value: number }) {
  return (
    <div
      style={{
        border: "1px solid #e5e7eb",
        borderRadius: 8,
        padding: 12
      }}
    >
      <div style={{ fontSize: 12, color: "#6b7280" }}>{title}</div>
      <div style={{ fontSize: 20, fontWeight: 600 }}>{value}</div>
    </div>
  );
}
