"use client";

import { useEffect, useState } from "react";
import { api, withAuth } from "@/lib/apiClient";
import { getAdminToken } from "@/lib/auth";

type Dispute = {
  id: string;
  bookingId: string;
  status: string;
  reason: string;
  adminNotes?: string | null;
  createdAt: string;
  booking?: {
    property?: { title: string };
    guest?: { fullName: string };
    host?: { fullName: string };
  };
};

export default function DisputesPage() {
  const [disputes, setDisputes] = useState<Dispute[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  const load = async () => {
    try {
      setLoading(true);
      setError(null);
      const token = getAdminToken();
      if (!token) {
        window.location.href = "/login";
        return;
      }
      withAuth();
      const res = await api.get("/admin/disputes");
      setDisputes(res.data);
    } catch {
      setError("Failed to load disputes");
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    load();
  }, []);

  return (
    <div>
      <h1>Disputes</h1>
      {error && <div style={{ color: "red" }}>{error}</div>}
      {loading ? (
        <div>Loading...</div>
      ) : disputes.length === 0 ? (
        <div>No disputes found.</div>
      ) : (
        <table style={{ marginTop: 16 }}>
          <thead>
            <tr>
              <th>ID</th>
              <th>Booking</th>
              <th>Parties</th>
              <th>Status</th>
              <th>Reason</th>
              <th>Admin Notes</th>
              <th>Created</th>
            </tr>
          </thead>
          <tbody>
            {disputes.map((d) => (
              <tr key={d.id}>
                <td>{d.id.slice(0, 8)}</td>
                <td>{d.bookingId.slice(0, 8)}</td>
                <td>
                  {d.booking?.guest?.fullName} vs {d.booking?.host?.fullName}
                  <br />
                  <span style={{ fontSize: 12, color: "#6b7280" }}>
                    {d.booking?.property?.title}
                  </span>
                </td>
                <td>{d.status}</td>
                <td>{d.reason}</td>
                <td>{d.adminNotes}</td>
                <td>{new Date(d.createdAt).toLocaleString()}</td>
              </tr>
            ))}
          </tbody>
        </table>
      )}
    </div>
  );
}
