"use client";

import { useEffect, useState } from "react";
import { api, withAuth } from "@/lib/apiClient";
import { getAdminToken } from "@/lib/auth";

type Booking = {
  id: string;
  status: string;
  checkInDate: string;
  checkOutDate: string;
  guestsCount: number;
  finalAmountNpr: number;
  guest?: { fullName: string; email: string };
  host?: { fullName: string; email: string };
  property?: { title: string; city: string };
};

export default function BookingsPage() {
  const [bookings, setBookings] = useState<Booking[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  const load = async () => {
    try {
      setLoading(true);
      setError(null);
      const token = getAdminToken();
      if (!token) {
        window.location.href = "/login";
        return;
      }
      withAuth();
      const res = await api.get("/admin/bookings");
      setBookings(res.data);
    } catch {
      setError("Failed to load bookings");
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    load();
  }, []);

  return (
    <div>
      <h1>Bookings</h1>
      {error && <div style={{ color: "red" }}>{error}</div>}
      {loading ? (
        <div>Loading...</div>
      ) : bookings.length === 0 ? (
        <div>No bookings found.</div>
      ) : (
        <table style={{ marginTop: 16 }}>
          <thead>
            <tr>
              <th>ID</th>
              <th>Property</th>
              <th>Guest</th>
              <th>Host</th>
              <th>Dates</th>
              <th>Status</th>
              <th>Amount (NPR)</th>
            </tr>
          </thead>
          <tbody>
            {bookings.map((b) => (
              <tr key={b.id}>
                <td>{b.id.slice(0, 8)}</td>
                <td>
                  {b.property?.title}
                  <br />
                  <span style={{ fontSize: 12, color: "#6b7280" }}>
                    {b.property?.city}
                  </span>
                </td>
                <td>
                  {b.guest?.fullName}
                  <br />
                  <span style={{ fontSize: 12, color: "#6b7280" }}>
                    {b.guest?.email}
                  </span>
                </td>
                <td>
                  {b.host?.fullName}
                  <br />
                  <span style={{ fontSize: 12, color: "#6b7280" }}>
                    {b.host?.email}
                  </span>
                </td>
                <td>
                  {new Date(b.checkInDate).toLocaleDateString()} →{" "}
                  {new Date(b.checkOutDate).toLocaleDateString()}
                </td>
                <td>{b.status}</td>
                <td>{b.finalAmountNpr}</td>
              </tr>
            ))}
          </tbody>
        </table>
      )}
    </div>
  );
}
